#include <sys/stat.h>

#include <cstdlib>

#include <fstream>
#include <sstream>

#include "ldastoolsal/autoarray.hh"
#include "ldastoolsal/Exec.hh"
#include "ldastoolsal/fstream.hh"
#include "ldastoolsal/unittest.h"

#include "framecpp/Common/BaseMemoryBuffer.hh"
#include "framecpp/Common/DynamicMemoryBuffer.hh"
#include "framecpp/Common/MemoryBuffer.hh"
#include "framecpp/Common/FrameBuffer.hh"

#include "framecpp/IFrameStream.hh"
#include "framecpp/OFrameStream.hh"
#include "framecpp/FrameH.hh"
#include "framecpp/FrRawData.hh"
#include "framecpp/FrVect.hh"

#define SMALL_FRAMES_TYPE 1

using namespace FrameCPP;
using LDASTools::AL::AutoArray;
using LDASTools::AL::DynamicPointerCast;
using LDASTools::AL::Exec;
using LDASTools::AL::GPSTime;
using LDASTools::AL::SharedPtr;

using FrameCPP::Common::FrameBuffer;
using FrameCPP::Common::MemoryBuffer;
using FrameCPP::Common::DynamicMemoryBuffer;
using FrameCPP::Common::ReadOnlyMemoryBuffer;
typedef FrameCPP::Common::FrameBufferInterface::Scanner Scanner;
using FrameCPP::FrameH;
using FrameCPP::FrRawData;
using FrameCPP::FrAdcData;

typedef FrameBuffer< LDASTools::AL::filebuf > frame_buffer_type;
typedef SharedPtr< FrameCPP::Common::FrameH > build_frame_type;
typedef FrameCPP::Common::FrameSpec::version_type frame_version_type;
typedef GPSTime start_time_type;
typedef REAL_8 dt_type;

LDASTools::Testing::UnitTest	Test;
std::string			TEST_DIR;

static build_frame_type create_frame( frame_version_type Version,
				      start_time_type Start, dt_type Dt );

//-----------------------------------------------------------------------
// Common tasks to be performed when exiting.
//-----------------------------------------------------------------------
inline void
depart( int ExitCode )
{
  exit( ExitCode );
}

inline const std::string&
filename_source( int Spec = FRAME_SPEC_CURRENT )
{
  static std::string filename;

  std::ostringstream filename_oss;

  if ( TEST_DIR.size( ) > 0 )
  {
    filename_oss << TEST_DIR << "/";
  }
  filename_oss << "Z-R_std_test_frame_ver" << Spec
	       << "-" << 600000000
	       << "-" << 1
	       << ".gwf"
    ;
  
  filename = filename_oss.str( );

  return filename;
}

void
test_dynamic_memory_buffer( )
{
  for ( INT_4U file_source_version = FRAME_SPEC_MIN,
	  file_source_version_end = FRAME_SPEC_MAX;
	file_source_version <= file_source_version_end;
	++file_source_version )
  {
    if ( file_source_version == 5 )
    {
      continue;
    }

    std::ostringstream	header;

    header << "test_dynamic_memory_buffer:"
	   << " file source version: " << file_source_version
      ;

    try
    {
      const std::string
	filename( filename_source( file_source_version ) );

      struct stat		stat_buf;
  
      stat( filename.c_str( ), &stat_buf );

      DynamicMemoryBuffer		mb;
      AutoArray< char >	read_buffer;
      DynamicMemoryBuffer::size_type	read_buffer_size = 0;
      DynamicMemoryBuffer::size_type	read_size = 0;
      std::ifstream			s;

      mb.Reset( );

      s.open( filename.c_str( ) );
      while ( ! mb.Ready( ) )
      {
	read_size = mb.NextBlockSize( );
	if ( read_buffer_size < read_size )
	{
	  read_buffer.reset( new char[ read_size ] );
	  read_buffer_size = read_size;
	}
	s.read( read_buffer.get( ), read_size );
	mb.NextBlock( read_buffer.get( ), read_size );
      }
      s.close( );

      IFrameStream		ifs( false, &mb );

      while( 1 )
      {
	try
	{
	  IFrameStream::frame_h_type	frame( ifs.ReadNextFrame( ) );

	  if ( ! frame )
	  {
	    break;
	  }

	  FrameH::rawData_type		rd( frame->GetRawData( ) );

	  if ( rd )
	  {
	    Test.Check(  rd->RefFirstAdc( ).size( ) > 0 )
	      << header.str( )
	      << ": firstAdc count: " <<  rd->RefFirstAdc( ).size( )
	      << std::endl
	      ;
	  }

	  Test.Check(  frame->RefProcData( ).size( ) > 0 )
	    << header.str( )
	    << ": FrProcData count: "
	    << frame->RefProcData( ).size( )
	    << std::endl
	    ;
	}
	catch( const std::out_of_range& E )
	{
	  break;
	}
      }
    }
    catch ( const std::exception& E )
    {
      Test.Check( false )
	<< header.str( )
	<< ": Caught an exception: " << E.what( )
	<< std::endl
	;
    }
    catch ( ... )
    {
      Test.Check( false )
	<< header.str( )
	<< ": Caught an unknown exception"
	<< std::endl
	;
    }
  }
}

void
test_memory_buffer( )
{
  for ( INT_4U file_source_version = FRAME_SPEC_MIN,
	  file_source_version_end = FRAME_SPEC_MAX;
	file_source_version <= file_source_version_end;
	++file_source_version )
  {
    if ( file_source_version == 5 )
    {
      continue;
    }

    std::ostringstream	header;

    header << "test_memory_buffer:"
	   << " file source version: " << file_source_version
      ;

    try
    {
      const std::string
	filename( filename_source( file_source_version ) );

      struct stat		stat_buf;
  
      stat( filename.c_str( ), &stat_buf );

      AutoArray< char >
	tbuf( new char[ stat_buf.st_size ] );
      std::ifstream	s;
      s.open( filename.c_str( ) );
      s.read( tbuf.get( ), stat_buf.st_size );
      AutoArray< char >
	buf( new char[ stat_buf.st_size ] );
      MemoryBuffer		mb( std::ios::in | std::ios::binary );
      mb.pubsetbuf( buf.get( ), stat_buf.st_size );
      mb.str( std::string( tbuf.get( ), stat_buf.st_size ) );
      s.close( );


      IFrameStream		ifs( false, &mb );

      while( 1 )
      {
	try
	{
	  IFrameStream::frame_h_type	frame( ifs.ReadNextFrame( ) );

	  if ( ! frame )
	  {
	    break;
	  }

	  FrameH::rawData_type		rd( frame->GetRawData( ) );

	  if ( rd )
	  {
	    Test.Check(  rd->RefFirstAdc( ).size( ) > 0 )
	      << header.str( )
	      << ": firstAdc count: " <<  rd->RefFirstAdc( ).size( )
	      << std::endl
	      ;
	  }

	  Test.Check(  frame->RefProcData( ).size( ) > 0 )
	    << header.str( )
	    << ": FrProcData count: "
	    << frame->RefProcData( ).size( )
	    << std::endl
	    ;
	}
	catch( const std::out_of_range& E )
	{
	  break;
	}
      }
    }
    catch ( const std::exception& E )
    {
      Test.Check( false )
	<< header.str( )
	<< ": Caught an exception: " << E.what( )
	<< std::endl
	;
    }
    catch ( ... )
    {
      Test.Check( false )
	<< header.str( )
	<< ": Caught an unknown exception"
	<< std::endl
	;
    }
  }
}

void
test_memory_buffer_ro( )
{
  for ( INT_4U file_source_version = FRAME_SPEC_MIN,
	  file_source_version_end = FRAME_SPEC_MAX;
	file_source_version <= file_source_version_end;
	++file_source_version )
  {
    if ( file_source_version == 5 )
    {
      continue;
    }

    std::ostringstream	header;

    header << "test_memory_buffer_ro:"
	   << " file source version: " << file_source_version
      ;

    try
    {
      const std::string
	filename( filename_source( file_source_version ) );

      struct stat		stat_buf;
  
      stat( filename.c_str( ), &stat_buf );

      AutoArray< char >
	buf( new char[ stat_buf.st_size ] );
      std::ifstream	s;
      s.open( filename.c_str( ) );
      s.read( buf.get( ), stat_buf.st_size );
      s.close( );

      ReadOnlyMemoryBuffer	mb;

      mb.pubsetbuf( buf.get( ), stat_buf.st_size );

      Test.Message( 20 )
	<< header.str( )
	<< " starting address: " << (void*)( buf.get( ) )
	<< " size: " << stat_buf.st_size
	<< " availablle: " << mb.in_avail( )
	<< " current: " << mb.pubseekpos( 0, std::ios_base::in )
	<< " current out: " << mb.pubseekpos( 0, std::ios_base::out )
	<< " current end: " << mb.pubseekoff( 0, std::ios_base::end, std::ios_base::in )
	<< std::endl
	;
      
      mb.pubseekpos( 0, std::ios_base::in );

      IFrameStream		ifs( false, &mb );

      while( 1 )
      {
	try
	{
	  IFrameStream::frame_h_type	frame( ifs.ReadNextFrame( ) );

	  if ( ! frame )
	  {
	    break;
	  }

	  FrameH::rawData_type		rd( frame->GetRawData( ) );

	  if ( rd )
	  {
	    Test.Check(  rd->RefFirstAdc( ).size( ) > 0 )
	      << header.str( )
	      << ": firstAdc count: " <<  rd->RefFirstAdc( ).size( )
	      << std::endl
	      ;
	  }

	  Test.Check(  frame->RefProcData( ).size( ) > 0 )
	    << header.str( )
	    << ": FrProcData count: "
	    << frame->RefProcData( ).size( )
	    << std::endl
	    ;
	}
	catch( const std::out_of_range& E )
	{
	  break;
	}
      }
    }
    catch ( const std::exception& E )
    {
      Test.Check( false )
	<< header.str( )
	<< ": Caught an exception: " << E.what( )
	<< std::endl
	;
    }
    catch ( ... )
    {
      Test.Check( false )
	<< header.str( )
	<< ": Caught an unknown exception"
	<< std::endl
	;
    }
  }
}

void
test_scanner( )
{
  for ( INT_4U file_source_version = FRAME_SPEC_MIN,
	  file_source_version_end = FRAME_SPEC_MAX;
	file_source_version <= file_source_version_end;
	++file_source_version )
  {
    if ( file_source_version == 5 )
    {
      continue;
    }

    std::ostringstream	header;

    header << "test_scanner:"
	   << " file source version: " << file_source_version
      ;

    try
    {
      const std::string
	filename( filename_source( file_source_version ) );

      struct stat	stat_buf;
  
      stat( filename.c_str( ), &stat_buf );

      std::ifstream			s;
      AutoArray< char >	read_buffer;
      Scanner::size_type		read_buffer_size = 0;
      Scanner::size_type		read_size = 0;
      Scanner				scanner;

      s.open( filename.c_str( ) );
      while ( ! scanner.Ready( ) )
      {
	read_size = scanner.NextBlockSize( );
	if ( read_buffer_size < read_size )
	{
	  read_buffer.reset( new char[ read_size ] );
	  read_buffer_size = read_size;
	}
	s.read( read_buffer.get( ), read_size );
	scanner.NextBlock( read_buffer.get( ), read_size );
      }
      s.close( );

      //-----------------------------------------------------------------
      // Display information about the stream
      //-----------------------------------------------------------------
      Test.Message( 10 )
	<< header.str( )
	<< " GPS Start: " << scanner.StartTime( )
	<< " DT: " << scanner.DeltaT( )
	<< std::endl
	;

    }
    catch ( const std::exception& E )
    {
      Test.Check( false )
	<< header.str( )
	<< ": Caught an exception: " << E.what( )
	<< std::endl
	;
    }
    catch ( ... )
    {
      Test.Check( false )
	<< header.str( )
	<< ": Caught an unknown exception"
	<< std::endl
	;
    }
  }
}

void
test_standard( )
{
  try
  {
    //-------------------------------------------------------------------
    // input buffer needed
    //-------------------------------------------------------------------
    std::unique_ptr< frame_buffer_type >
      i_frame_buf( new frame_buffer_type( std::ios::in ) );

    std::unique_ptr< frame_buffer_type >
      o_frame_buf( new frame_buffer_type( std::ios::out ) );

    //-------------------------------------------------------------------
    // Calculate the filename
    //-------------------------------------------------------------------
    std::ostringstream	filename;

    if ( TEST_DIR.size( ) > 0 )
    {
      filename << TEST_DIR << "/";
    }
    filename << "Z-R_std_test_frame_ver" << FRAME_SPEC_CURRENT
	     << "-" << 600000000
	     << "-" << 1
	     << ".gwf"
      ;

    std::ostringstream	ofilename;

    if ( TEST_DIR.size( ) > 0 )
    {
      ofilename << TEST_DIR << "/";
    }
    ofilename << "Z-R_std_test_frame_mem_buf_ver" << FRAME_SPEC_CURRENT
	      << "-" << 600000000
	      << "-" << 1
	      << ".gwf"
      ;
    //-----------------------------------------------------------------
    // Open the buffer and the stream
    //-----------------------------------------------------------------
    MemoryBuffer	omb( std::ios::out );
    i_frame_buf->open( filename.str( ).c_str( ),
		       std::ios::in | std::ios::binary );

    o_frame_buf->open( ofilename.str( ).c_str( ),
		       std::ios::out | std::ios::binary );

    IFrameStream	ifs( i_frame_buf.release( ) );

    OFrameStream	ofs_mem( false, &omb );

    try
    {
      while( 1 )
      {
	IFrameStream::frame_h_type	f = ifs.ReadNextFrame( );
		   
	ofs_mem.WriteFrame( f );
      }
    }
    catch( const std::out_of_range& E )
    {
      ofs_mem.Close( );
      omb.pubsync( );

      OFrameStream	ofs( o_frame_buf.release( ) );

      //-----------------------------------------------------------------
      // load the input buffer with the output from the previous read
      //-----------------------------------------------------------------
      MemoryBuffer	imb( std::ios::in );
      imb.str( omb.str( ) );

      IFrameStream	ifs_mem( false, &imb );

      try
      {
	while( 1 )
	{
	  IFrameStream::frame_h_type	f = ifs_mem.ReadNextFrame( );
	  ofs.WriteFrame( f );
	}
      }
      catch( const std::out_of_range& E )
      {
	// Do Nothing
      }
      ofs.Close( );
    }
    catch( std::exception& Exception )
    {
      throw;
    }

  }
  catch ( const std::exception& E )
  {
    Test.Check( false ) << "Caught an exception: " << E.what( ) << std::endl;
  }
  catch ( ... )
  {
    Test.Check( false ) << "Caught an unknown exception" << std::endl;
  }
}

void
test_small_frames( )
{
  typedef std::list< INT_4U > frames_per_file_type;

  build_frame_type	frame;
  frames_per_file_type	frames_per_file;

  frames_per_file.push_back( 50 );
  frames_per_file.push_back( 60 );


  for ( frame_version_type
	  cur = 8,
	  last = 8 + 1;
	cur != last;
	++cur )
  {
    for ( frames_per_file_type::const_iterator
	    cur_fpf = frames_per_file.begin( ),
	    last_fpf = frames_per_file.end( );
	  cur_fpf != last_fpf;
	  ++cur_fpf )
    {
      start_time_type	start( 913423140, 0 );
      dt_type		dt( 1 );

      std::ostringstream	filename;
      filename << "Z-SmallFrame_" << *cur_fpf
	       << "-" << start.GetSeconds( )
	       << "-" << int( dt * *cur_fpf )
	       << ".gwf"
	;
#if SMALL_FRAMES_TYPE == 0
      INT_4U	OBUFFER_SIZE = ( 10 * 1024 * 1024 );
      AutoArray< frame_buffer_type::char_type  >
	  
	obuffer( new frame_buffer_type::char_type[ OBUFFER_SIZE ] );
      std::unique_ptr< frame_buffer_type >
	obuf( new frame_buffer_type( std::ios::out ) );
      obuf->pubsetbuf( obuffer.get( ), OBUFFER_SIZE );
      obuf->open( filename.str( ).c_str( ),
		  std::ios::out | std::ios::binary );

      OFrameStream	ofs( obuf.release( ) );
#elif SMALL_FRAMES_TYPE == 1
      INT_4U	OBUFFER_SIZE = ( 10 * 1024 * 1024 );
      AutoArray< frame_buffer_type::char_type  >
	  
	obuffer( new frame_buffer_type::char_type[ OBUFFER_SIZE ] );
      FrameCPP::Common::MemoryBuffer
	*obuf( new FrameCPP::Common::MemoryBuffer( std::ios::out ) );
      obuf->pubsetbuf( obuffer.get( ), OBUFFER_SIZE);
      FrameCPP::OFrameStream	ofs( obuf );
#endif
      ofs.DisableTOC( );

      for ( frames_per_file_type::value_type
	      cur_sec = 0,
	      last_sec = *cur_fpf;
	    cur_sec != last_sec;
	    ++cur_sec,
	    start += dt )
      {
	frame = create_frame( cur, start, dt );
	if ( frame )
	{
	  ofs.WriteFrame( frame,
			  FrameCPP::FrVect::RAW,
			  FrameCPP::FrVect::DEFAULT_GZIP_LEVEL );
	}
      }
      ofs.Close( );
#if SMALL_FRAMES_TYPE == 1
      std::ofstream	fstr;
      fstr.open( filename.str( ).c_str( ),
		  std::ios::out | std::ios::binary );
      fstr.write( obuf->str( ).c_str( ),
		  obuf->str( ).length( ) );
      fstr.close( );
#endif /* 0 */
      char cfilename[4096];

      ::strcpy( cfilename, filename.str( ).c_str( ) );

      Exec cmd( "./framecpp_verify",
		"--verbose",
		"--verbose",
		cfilename,
		Exec::COMMAND_END );
      cmd.Spawn( );
      Test.Check( cmd.ExitCode( ) == 0 )
	<< "Verify gstreamer short frames: "
	<< filename.str( )
	<< std::endl
	;
      if ( cmd.ExitCode( ) != 0 )
      {
	Test.Message( 10 )
	  << "StdOut: " << cmd.Stdout( )
	  << std::endl
	  << "StdErr:" << cmd.Stderr( )
	  << std::endl
	  ;
      }

    }
  }
}

void
test_read_events( )
{
  try
  {
    static const size_t EVENTS = 2;
    const std::string event_name( "TesT" );

    //-------------------------------------------------------------------
    // input buffer needed
    //-------------------------------------------------------------------
    std::unique_ptr< frame_buffer_type >
      i_frame_buf( new frame_buffer_type( std::ios::in ) );

    //-------------------------------------------------------------------
    // Calculate the filename
    //-------------------------------------------------------------------
    std::ostringstream	filename;

    if ( TEST_DIR.size( ) > 0 )
    {
      filename << TEST_DIR << "/";
    }
    filename << "Z-R_std_test_frame_ver" << FRAME_SPEC_CURRENT
	     << "-" << 600000000
	     << "-" << 1
	     << ".gwf"
      ;

    //-----------------------------------------------------------------
    // Open the buffer and the stream
    //-----------------------------------------------------------------
    i_frame_buf->open( filename.str( ).c_str( ),
		       std::ios::in | std::ios::binary );

    IFrameStream	ifs( i_frame_buf.release( ) );

    size_t x = 0;
    try
    {
      while ( true )
      {
	ifs.ReadFrEvent( event_name, x );
	++x;
      }
    }
    catch( const std::range_error& Error )
    {
    }
    catch( const std::exception& Error )
    {
    }
    Test.Check( EVENTS == x )
      << "Verifying the expected number of events"
      << std::endl
      ;

    {
      static const size_t SIM_EVENTS = size_t( 1 );
      const std::string sim_event_name( "TesT" );
      x = 0;
      try
      {
	while ( true )
        {
	  ifs.ReadFrSimEvent( sim_event_name, x );
	  ++x;
	}
      }
      catch( const std::range_error& Error )
      {
      }
      catch( const std::exception& Error )
      {
      }
      Test.Check( SIM_EVENTS == x )
	<< "Verifying the expected number of simulated events"
	<< "(expected: " << SIM_EVENTS << " received: " << x << ")"
	<< std::endl
	;
    }
  }
  catch( std::exception& Exception )
  {
    Test.Check( false ) << "Caught an exception: " << Exception.what( ) << std::endl;
  }
}

int
main( int ArgC, char**ArgV )
{
  //---------------------------------------------------------------------
  // Initialize the testing framework
  //---------------------------------------------------------------------
  Test.Init( ArgC, ArgV );

  const char* td( ::getenv( "TEST_DIR" ) );
  if ( td )
  {
    TEST_DIR = td;
  }

  FrameCPP::Initialize( );

  //---------------------------------------------------------------------
  //
  //---------------------------------------------------------------------
  test_standard( );
  test_memory_buffer( );
  test_dynamic_memory_buffer( );
  test_memory_buffer_ro( );
  test_scanner( );
  test_small_frames( );
  test_read_events( );

  //---------------------------------------------------------------------
  // Testing is complete. Report back to the user
  //---------------------------------------------------------------------

  Test.Exit();
}

template< frame_version_type > build_frame_type
frameh( start_time_type StartTime, dt_type Dt );

template<> build_frame_type
frameh< 8 >( start_time_type StartTime, dt_type Dt )
{
#undef FH_VERSION
#define FH_VERSION Version_8

  SharedPtr< FrameH >
    frameh( new FH_VERSION :: FrameH( "SmallFrame",
				      0,
				      0,
				      StartTime,
				      StartTime.GetLeapSeconds( ),
				      Dt ) );

  std::string			name( "SFProc" );
  std::vector< REAL_8 >		d( 512 );
  REAL_8			value( StartTime.GetSeconds( ) );

  for ( std::vector< REAL_8 >::iterator
	  cur_d = d.begin( ),
	  last_d = d.end( );
	cur_d != last_d;
	++cur_d,
	value += Dt )
  {
    *cur_d = value;
  }
  FH_VERSION :: Dimension 	dim( d.size( ) );
  FH_VERSION :: FrVect		v( name, 1, &dim, &d[ 0 ] );
  FH_VERSION :: FrProcData
    proc( name, "",
	  FrProcData::TIME_SERIES, FrProcData::UNKNOWN_SUB_TYPE,
	  0, /* TimeOffset */
	  0,/* TRange */
	  0, /* FShift */
	  0, /* Phase */
	  0, /* FRange */
	  0 /* BW */
	  );
  proc.RefData( ).append( v );
  frameh->RefProcData( ).append( proc );

  return DynamicPointerCast< build_frame_type::element_type >( frameh );
#undef FH_VERSION
}

static build_frame_type
create_frame( frame_version_type Version,
	      start_time_type Start, dt_type Dt )
{
  build_frame_type	result;

  switch( Version )
  {
  case 8:
    result = frameh< 8 >( Start, Dt );
    break;
  default:
    break;
  }
  return result;
}
